/************************************************************************************************/
/*																								*/
/* FILE NAME   : Display.cpp								VERSION : 							*/
/*																								*/
/* DESCRIPTION : ﾃﾞｨｽﾌﾟﾚｲ ｿｰｽﾌｧｲﾙ																*/
/*																								*/
/* HISTORY     :																				*/
/*																								*/
/************************************************************************************************/

/************************************************************************************************/
/*	《１》取込みファイル定義																	*/
/************************************************************************************************/
#include "Display.h"
#include "Fonts.h"
#include <Wire.h>
#include <SPI.h>

/************************************************************************************************/
/*	《２》ファイル内でのみ使用する定数定義														*/
/************************************************************************************************/
/*************************/
/*         GPIO          */
/*************************/
#define GPIO_LOW				0
#define GPIO_HIGH				1

#define M_SPI_SCK				18														// SPIﾎﾟｰﾄ
#define M_SPI_DOUT				19
#define M_SPI_DIN				23
#define M_SPI_CS				5
#define M_SPI_DC				25
#define M_SPI_RST				26
#define M_SPI_BUSY				27
#define M_I2C_SDA				21														// I2Cﾎﾟｰﾄ
#define M_I2C_SCL				22
#define M_I2C_ADDRESS			0x3c
#define M_ESP_RXD				16														// UART
#define M_ESP_TXD				17

/*************************/
/*        その他         */
/*************************/
// VSPIとHSPIの差はないらしいが詳細は不明(SSはｽﾚｰﾌﾞｾﾚｸﾄの意味)
// HSPI - SCK,MISO,MOSI,SS = 14,12,13,15
// VSPI - SCK,MISO,MOSI,SS = 18,19,23,5
//
// Mode	      CLK初期値	出力CLKｴｯｼﾞ 読出しCLKｴｯｼﾞ
// SPI_MODE0	0	    立ち下がり	立ち上がり
// SPI_MODE1	1	    立ち上がり	立ち下がり
// SPI_MODE2	0	    立ち上がり	立ち下がり
// SPI_MODE3	1	    立ち下がり	立ち上がり
//
#define _C_SIZE_OLED_WIDTH		128
#define _C_SIZE_OLED_HEIGHT		64
#define _C_SIZE_OLED_PAGE		8
#define _C_SIZE_E_PAPER_WIDTH	122
#define _C_SIZE_E_PAPER_HEIGHT	250

#define _C_TIM_WAIT_2			2
#define _C_TIM_WAIT_10			10
#define _C_TIM_WAIT_20			20
#define _C_TIM_WAIT_100			100

#define _C_SIZE_NUM_TO_STR		255

#define _C_DAT_COLOR_WHITE		0xff													// 白色
#define _C_DAT_COLOR_BLACK		0x00													// 黒色

/************************************************************************************************/
/*	《３》ファイル内でのみ使用するマクロ定義													*/
/************************************************************************************************/

/************************************************************************************************/
/*	《４》ファイル内でのみ使用するTypedef定義(型定義)											*/
/************************************************************************************************/

/************************************************************************************************/
/*	《５》ファイル内でのみ使用する構造体/共用体定義												*/
/************************************************************************************************/

/************************************************************************************************/
/*	《６》ファイル内で共有する変数定義															*/
/************************************************************************************************/
DISPLAY_TYPE		_gDisplay         = enOled;											// ﾃﾞｨｽﾌﾟﾚｲﾀｲﾌﾟ
PAINT				_gPaint           = { 0 };											// ｲﾒｰｼﾞ構造体変数
uint16_t			_gBackColor       = _C_DAT_COLOR_WHITE;								// 背景色
uint16_t			_gDrawColor       = _C_DAT_COLOR_BLACK;								// 描画色

SPIClass			_gSpi( VSPI );														// SPI通信ｲﾝｽﾀﾝｽ

uint8_t WF_PARTIAL_2IN13_V3[ 159 ] =
{
	0x00,0x40,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x80,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x40,0x40,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x14,0x00,0x00,0x00,0x00,0x00,0x00,  
	0x01,0x00,0x00,0x00,0x00,0x00,0x00,
	0x01,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x22,0x22,0x22,0x22,0x22,0x22,0x00,0x00,0x00,
	0x22,0x17,0x41,0x00,0x32,0x36,
};

uint8_t WS_20_30_2IN13_V3[ 159 ] =
{											
	0x80,	0x4A,	0x40,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,
	0x40,	0x4A,	0x80,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,
	0x80,	0x4A,	0x40,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,
	0x40,	0x4A,	0x80,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,
	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,
	0x0f,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,					
	0x0f,	0x00,	0x00,	0x0f,	0x00,	0x00,	0x02,					
	0x0f,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,					
	0x01,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,					
	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,					
	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,					
	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,					
	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,					
	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,					
	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,					
	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,					
	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,					
	0x22,	0x22,	0x22,	0x22,	0x22,	0x22,	0x00,	0x00,	0x00,			
	0x22,	0x17,	0x41,	0x00,	0x32,	0x36						
};

/************************************************************************************************/
/*	《７》ファイル内で共有する関数プロトタイプ宣言												*/
/************************************************************************************************/
void _Oled_Config();
void _Oled_InitDisplay();
void _Oled_ClearDisplay();
void _Oled_Display( uint8_t* Image );
uint8_t _Oled_GetData( uint8_t* Image, uint8_t Page, uint8_t Block, uint8_t X );

void _Epaper_Config();
void _Epaper_InitDisplay();
void _Epaper_Reset();
void _Epaper_Sleep();
void _Epape_WaitBusy();
void _Epaper_SetWindows( uint16_t Sx, uint16_t Sy, uint16_t Ex, uint16_t Ey );
void _Epaper_SetLutByHost( uint8_t* Lut );
void _Epaper_SetLut( uint8_t* Lut );
void _Epaper_SetCursor( uint16_t Sx, uint16_t Sy );
void _Epaper_TurnOnDisplay();
void _Epaper_TurnOnDisplayPartial();
void _Epaper_ClearDisplay();
void _Epaper_Display( uint8_t* Image );
void _Epaper_Display_Base( uint8_t* Image );
void _Epaper_Display_Partial( uint8_t* Image );
void _Epaper_CommandWrite( uint8_t Data );
void _Epaper_DataWrite( uint8_t Data );
void _Epaper_CommandWriteBytes( uint8_t *Data, uint16_t Count );
void _Epaper_DataWriteBytes(uint8_t *Data, uint16_t Count );

/************************************************************************************************/
/*	《８》ＯＳ資源定義																			*/
/************************************************************************************************/

/************************************************************************************************/
/*	《９》外部参照変数定義(ドメイン・グローバルな外部変数定義)									*/
/************************************************************************************************/
/*************************/
/*        初期化         */
/*************************/
void Display_Init( DISPLAY_TYPE Type )
{
	_gDisplay = Type;																	// ﾃﾞｨｽﾌﾟﾚｲﾀｲﾌﾟ

	/*************************/
	/*  ﾎﾟｰﾄ設定と領域取得   */
	/*************************/
	uint16_t width;
	uint16_t height;
	uint16_t rotate;
	switch( _gDisplay )
	{
		case enOled:																	// OLED
			_Oled_Config();
			width  = _C_SIZE_OLED_WIDTH;
			height = _C_SIZE_OLED_HEIGHT;
			rotate = C_DAT_ROTATE_180;
			break;
		case enEpaper:																	// 電子ﾍﾟｰﾊﾟｰ
			_Epaper_Config();
			width  = _C_SIZE_E_PAPER_WIDTH;
			height = _C_SIZE_E_PAPER_HEIGHT;
			rotate = C_DAT_ROTATE_90;
			break;
		default:
			return;
	}

	/*************************/
	/*  ｲﾒｰｼﾞ構造体変数ｾｯﾄ   */
	/*************************/
	/* ---< 領域解放 >--- */
	if( _gPaint.Image != NULL )
	{
		free( _gPaint.Image );
	}
	_gPaint.Image = NULL;
	/* ---< ｲﾒｰｼﾞ構造体変数ｾｯﾄ >--- */
	uint16_t size = (( width % 8 == 0 )? ( width / 8 ): ( width / 8 + 1 )) * height;
	if(( _gPaint.Image = (uint8_t*)malloc( size )) == NULL )							// ｲﾒｰｼﾞ領域取得
	{
		return;
	}
	Paint_NewImage( _gPaint.Image, width, height, rotate );								// ｲﾒｰｼﾞ構造体変数ｾｯﾄ
	Paint_Clear( _gBackColor );															// ｲﾒｰｼﾞ領域ｸﾘｱ
}
/*************************/
/*        表示ｸﾘｱ        */
/*************************/
void Display_Clear()
{
	Paint_Clear( _gBackColor );															// ｲﾒｰｼﾞ領域ｸﾘｱ
	switch( _gDisplay )
	{
		case enOled:																	// OLED
			_Oled_InitDisplay();
			_Oled_ClearDisplay();
			break;
		case enEpaper:																	// 電子ﾍﾟｰﾊﾟｰ
			_Epaper_InitDisplay();
			_Epaper_ClearDisplay();
			break;
		default:
			return;
	}
}
/*************************/
/*      文字列描画       */
/*************************/
void Display_Draw( uint16_t Sx, uint16_t Sy, const char* pString, FONT* Font )
{
	switch( _gDisplay )
	{
		case enOled:																	// OLED
			Paint_DrawString( Sx, Sy, pString, Font, _gDrawColor, _gBackColor );
			_Oled_Display( _gPaint.Image );
			break;
		case enEpaper:																	// 電子ﾍﾟｰﾊﾟｰ
			Paint_DrawString( Sx, Sy, pString, Font, _gDrawColor, _gBackColor );
			_Epaper_Display( _gPaint.Image );
			break;
		default:
			return;
	}
}
/*************************/
/*       数値描画        */
/*************************/
void Display_Draw( uint16_t Sx, uint16_t Sy, int32_t Nummber, FONT* Font )
{
	switch( _gDisplay )
	{
		case enOled:																	// OLED
			Paint_DrawNum( Sx, Sy, Nummber, Font, _gDrawColor, _gBackColor );
			_Oled_Display( _gPaint.Image );
			break;
		case enEpaper:																	// 電子ﾍﾟｰﾊﾟｰ
			Paint_DrawNum( Sx, Sy, Nummber, Font, _gDrawColor, _gBackColor );
			_Epaper_Display( _gPaint.Image );
			break;
		default:
			return;
	}
}
/*************************/
/*         ｽﾘｰﾌﾟ         */
/*************************/
void Display_Sleep()
{
	switch( _gDisplay )
	{
		case enOled:																	// OLED
			break;
		case enEpaper:																	// 電子ﾍﾟｰﾊﾟｰ
			_Epaper_Sleep();
			break;
		default:
			return;
	}
}
/*************************/
/*   表示ﾃﾞﾊﾞｲｽのｻｲｽﾞ    */
/*************************/
uint16_t Display_GetisplayWidth()
{
	switch( _gDisplay )
	{
		case enOled:																	// OLED
			return _C_SIZE_OLED_WIDTH;
		case enEpaper:																	// 電子ﾍﾟｰﾊﾟｰ
			return _C_SIZE_E_PAPER_WIDTH;
		default:
			return 0;
	}
}
uint16_t Display_GetDisplayHeight()
{
	switch( _gDisplay )
	{
		case enOled:																	// OLED
			return _C_SIZE_OLED_HEIGHT;
		case enEpaper:																	// 電子ﾍﾟｰﾊﾟｰ
			return _C_SIZE_E_PAPER_HEIGHT;
		default:
			return 0;
	}
}
/************************************************************************************************/
/*	機能概要：電子ﾍﾟｰﾊﾟｰ設定																	*/
/*	引数	：なし																				*/
/*	戻り値	：なし																				*/
/*	備考	：																					*/
/************************************************************************************************/
/**************** 送信ﾌﾟﾛﾄｺﾙ ****************/
// I2Cｱﾄﾞﾚｽ→0x80(1ﾊﾞｲﾄｺﾏﾝﾄﾞﾀｸﾞ)→ｺﾏﾝﾄﾞ→終了
// I2Cｱﾄﾞﾚｽ→0x00(複数ｺﾏﾝﾄﾞﾀｸﾞ )→ｺﾏﾝﾄﾞ→ｺﾏﾝﾄﾞ→ｺﾏﾝﾄﾞ→終了
// I2Cｱﾄﾞﾚｽ→0x00(複数ｺﾏﾝﾄﾞﾀｸﾞ )→ｺﾏﾝﾄﾞ→ﾃﾞｰﾀ →ﾃﾞｰﾀ →終了
// I2Cｱﾄﾞﾚｽ→0x40(ﾃﾞｰﾀﾀｸﾞ      )→ﾃﾞｰﾀ →ﾃﾞｰﾀ →ﾃﾞｰﾀ →終了
//
/***************** 画面定義 *****************/
// Page0～Page7は横127ﾋﾟｸｾﾙに対し、縦に8つ並ぶ
// 縦1Pageだけで言えば、1ﾊﾞｲﾄで縦8ﾋﾟｸｾﾙを表す
//
// -------------- 128ﾋﾟｸｾﾙ --------------
//        Page0
//   0   1   2   3   4   5   6        127 |
//  b0  b0  b0  b0  b0  b0  b0  ・     b0 |
//  b1  b1  b1  b1  b1  b1  b1	・	   b1 |
//  b2  b2  b2  b2  b2  b2  b2	・	   b2 |
//  b3  b3  b3  b3  b3  b3  b3	・	   b3 | 8ﾋﾟｸｾﾙ
//  b4  b4  b4  b4  b4  b4  b4	・	   b4 |
//  b5  b5  b5  b5  b5  b5  b5	・	   b5 |
//  b6  b6  b6  b6  b6  b6  b6	・	   b6 |
//  b7  b7  b7  b7  b7  b7  b7	・	   b7 |
//        Page1
//   0   1   2   3   4   5   6        127 |
//  b0  b0  b0  b0  b0  b0  b0  ・     b0 |
//  b1  b1  b1  b1  b1  b1  b1	・	   b1 |
//  b2  b2  b2  b2  b2  b2  b2	・	   b2 |
//  b3  b3  b3  b3  b3  b3  b3	・	   b3 | 8ﾋﾟｸｾﾙ   8 x 8 ﾋﾟｸｾﾙ 64ﾋﾟｸｾﾙ
//  b4  b4  b4  b4  b4  b4  b4	・	   b4 |
//  b5  b5  b5  b5  b5  b5  b5	・	   b5 |
//  b6  b6  b6  b6  b6  b6  b6	・	   b6 |
//  b7  b7  b7  b7  b7  b7  b7	・	   b7 |
//
//  ・  ・  ・   ・  ・  ・ ・  ・  ・ ・
//  ・  ・  ・   ・  ・  ・ ・  ・  ・ ・
//  ・  ・  ・   ・  ・  ・ ・  ・  ・ ・
//  ・  ・  ・   ・  ・  ・ ・  ・  ・ ・
//  ・  ・  ・   ・  ・  ・ ・  ・  ・ ・
//        Page7
//   0   1   2   3   4   5   6        127 |
//  b0  b0  b0  b0  b0  b0  b0  ・     b0 |
//  b1  b1  b1  b1  b1  b1  b1	・	   b1 |
//  b2  b2  b2  b2  b2  b2  b2	・	   b2 |
//  b3  b3  b3  b3  b3  b3  b3	・	   b3 | 8ﾋﾟｸｾﾙ
//  b4  b4  b4  b4  b4  b4  b4	・	   b4 |
//  b5  b5  b5  b5  b5  b5  b5	・	   b5 |
//  b6  b6  b6  b6  b6  b6  b6	・	   b6 |
//  b7  b7  b7  b7  b7  b7  b7	・	   b7 |
//
/************** ｱﾄﾞﾚｯｼﾝｸﾞﾓｰﾄﾞ ***************/
// 水平ｱﾄﾞﾚｯｼﾝｸﾞﾓｰﾄﾞでは、ﾃﾞｨｽﾌﾟﾚｲRAMの読み取り/書き込み後、列ｱﾄﾞﾚｽﾎﾟｲﾝﾀが自動的に1増加する
// 列ｱﾄﾞﾚｽﾎﾟｲﾝﾀが列終了ｱﾄﾞﾚｽに達すると、列ｱﾄﾞﾚｽﾎﾟｲﾝﾀが列開始ｱﾄﾞﾚｽにﾘｾｯﾄされ、ﾍﾟｰｼﾞｱﾄﾞﾚｽﾎﾟｲﾝﾀが増加する
// 列ｱﾄﾞﾚｽとﾍﾟｰｼﾞｱﾄﾞﾚｽの両方のﾎﾟｲﾝﾀが終了ｱﾄﾞﾚｽに到達すると、ﾎﾟｲﾝﾀは列の開始ｱﾄﾞﾚｽとﾍﾟｰｼﾞの開始ｱﾄﾞﾚｽ
// にﾘｾｯﾄされる 
//
// ﾍﾟｰｼﾞｱﾄﾞﾚｯｼﾝｸﾞﾓｰﾄﾞは、水平ｱﾄﾞﾚｯｼﾝｸﾞﾓｰﾄﾞでﾍﾟｰｼﾞｱﾄﾞﾚｽﾎﾟｲﾝﾀは変化しないのと同じ

void _Oled_Config()
{
	pinMode( M_I2C_SDA, INPUT_PULLUP );													// SDA
	pinMode( M_I2C_SCL, INPUT_PULLUP );													// SCL
	Wire.begin( M_I2C_SDA, M_I2C_SCL );
}
void _Oled_InitDisplay()
{
	const uint8_t config[] =
	{
		0xa8,			// 使用する行数													(0xa8)
		0x3f,			// 	設定値　64行
		0xd3,			// 画面垂直方向のｵﾌｾｯﾄ(ずらし)									(0xd3)
		0x00,			// 	ずらしなし　0x00
		0x40,			// 画面ｽﾀｰﾄﾗｲﾝ 全領域使用するので0x40							(0x40/0x7f)
		0xa1,			// 水平方向の反転　ﾃﾞﾌｫﾙﾄは右下ｽﾀｰﾄ、左上ｽﾀｰﾄ0x41				(0xa0/0xa1)
		0xc8,			// 垂直方向の反転　ﾃﾞﾌｫﾙﾄは右下ｽﾀｰﾄ、左上ｽﾀｰﾄ0xc8				(0xc0/0xc8)
		0xda,			// Set COM Pins Hardware Configuration							(0xda)
		0x12,			// 	値を変えても挙動に変化無し。ﾃﾞﾌｫﾙﾄ値0x12でOK
		0x81,			// 256諧調のｺﾝﾄﾗｽﾄ												(0x81)
		0x7f,			// 	ﾃﾞﾌｫﾙﾄ0x7f
		0xa4,			// 0xa4で画面ﾒﾓﾘの内容を画面表示、0xa5でﾃｽﾄ用(と思われる)		(0xa4/0xa5)
		0xa6,			// 白黒反転設定。通常は1が発光だが、その逆(0xa7)の設定も可能	(0xa6/0xa7)
		0xd5,			// ﾃﾞｨｽﾌﾟﾚｲｸﾛｯｸ設定												(0xd5)
		0x80,			// 	ﾃﾞﾌｫﾙﾄ0x80
		0x20,			// ﾒﾓﾘｱﾄﾞﾚｯｼﾝｸﾞﾓｰﾄﾞ												(0x20)
		0x00,			// 	0x00 水平ｱﾄﾞﾚｯｼﾝｸﾞ
						// 	0x01 垂直ｱﾄﾞﾚｯｼﾝｸﾞ
						// 	0x10 ﾍﾟｰｼﾞｱﾄﾞﾚｯｼﾝｸﾞ(ﾃﾞﾌｫﾙﾄ)
		0x8d,			// ﾃﾞｨｽﾌﾟﾚｲｵﾝとするﾁｬｰｼﾞﾎﾟﾝﾌﾟ									(0x8d)
		0x14,			// 	ﾁｬｰｼﾞﾎﾟﾝﾌﾟﾚｷﾞｭﾚｰﾀｵﾝ0x14
		0xaf			// 0xaeは画面ｵﾌ、0xafは画面ｵﾝ									(0xae/0xaf)
	};

	Wire.beginTransmission( M_I2C_ADDRESS );
	Wire.write( 0x00 );																	// 複数ｺﾏﾝﾄﾞﾀｸﾞ
	for( int index = 0; index < sizeof( config ); index++ )
	{
		Wire.write( config[ index ] );
	}
	Wire.endTransmission();
}
void _Oled_ClearDisplay()
{
	for( int page = 0; page < _C_SIZE_OLED_PAGE; page++ )
	{
		Wire.beginTransmission( M_I2C_ADDRESS );
		Wire.write( 0x80 );																// 1ﾊﾞｲﾄｺﾏﾝﾄﾞﾀｸﾞ
		Wire.write( 0xb0 | page );														// ﾍﾟｰｼﾞ指定
		Wire.write( 0x00 );																// 複数ｺﾏﾝﾄﾞﾀｸﾞ
		Wire.write( 0x21 );																// 開始列、終了列
		Wire.write( 0x00 );																// 開始列0
		Wire.write( 0x7f );																// 開始列127
		Wire.endTransmission();
		// 一旦連続送信の分割(Arduinoなんかは32ﾊﾞｲﾄまでらしい)
		// 8ﾊﾞｲﾄを16回送信すると128ﾊﾞｲﾄ
		for( int block = 0; block < 16; block++ )
		{
			Wire.beginTransmission( M_I2C_ADDRESS );
			Wire.write( 0x40 );															// ﾃﾞｰﾀﾀｸﾞ
			for( int x = 0; x < 8; x++ )
			{
				Wire.write( 0xff );														// 白塗り
//				Wire.write( 0x00 );														// 黒塗り
			}
			Wire.endTransmission();
		}
	}
}
void _Oled_Display( uint8_t* Image )
{
	uint16_t x = 0;
	uint16_t y = 0;
	uint16_t addr;
	for( uint16_t page = 0; page < _C_SIZE_OLED_PAGE; page++ )
	{
		Wire.beginTransmission( M_I2C_ADDRESS );
		Wire.write( 0x80 );																// 1ﾊﾞｲﾄｺﾏﾝﾄﾞﾀｸﾞ
		Wire.write( 0xb0 | page );														// ﾍﾟｰｼﾞ指定
		Wire.write( 0x00 );																// 複数ｺﾏﾝﾄﾞﾀｸﾞ
		Wire.write( 0x21 );																// 開始列、終了列
		Wire.write( 0x00 );																// 開始列0
		Wire.write( 0x7f );																// 開始列127
		Wire.endTransmission();
		// 一旦連続送信の分割(Arduinoなんかは32ﾊﾞｲﾄまでらしい)
		// 8ﾊﾞｲﾄを16回送信すると128ﾊﾞｲﾄ
		x = 0;
		for( uint16_t block = 0; block < 16; block++ )
		{
			Wire.beginTransmission( M_I2C_ADDRESS );
			Wire.write( 0x40 );															// ﾃﾞｰﾀﾀｸﾞ
			
			// ｲﾒｰｼﾞﾃﾞｰﾀ
			// 01100001
			// 10101000
			// 11100011
			// 01010111
			// 11111000
			// 01111111
			// 00000000
			// 11111111
			// ﾋﾞｯﾄ7列　10010110
			// ﾋﾞｯﾄ6列　10111101
			// ・・・・・・
			// ﾋﾞｯﾄ0列　10101101
			// ﾋﾞｯﾄ列7から順に8ﾊﾞｲﾄ送信する
			uint8_t data;
			for( int16_t bit = 7; bit >= 0; bit-- )
			{
				data = 0;
				addr = ( y + 0 ) * _gPaint.WidthByte + x;
				data |= ((( _gPaint.Image[ addr ] >> bit ) & 0x01 ) << 0 );

				addr = ( y + 1 ) * _gPaint.WidthByte + x;
				data |= ((( _gPaint.Image[ addr ] >> bit ) & 0x01 ) << 1 );

				addr = ( y + 2 ) * _gPaint.WidthByte + x;
				data |= ((( _gPaint.Image[ addr ] >> bit ) & 0x01 ) << 2 );

				addr = ( y + 3 ) * _gPaint.WidthByte + x;
				data |= ((( _gPaint.Image[ addr ] >> bit ) & 0x01 ) << 3 );

				addr = ( y + 4 ) * _gPaint.WidthByte + x;
				data |= ((( _gPaint.Image[ addr ] >> bit ) & 0x01 ) << 4 );

				addr = ( y + 5 ) * _gPaint.WidthByte + x;
				data |= ((( _gPaint.Image[ addr ] >> bit ) & 0x01 ) << 5 );

				addr = ( y + 6 ) * _gPaint.WidthByte + x;
				data |= ((( _gPaint.Image[ addr ] >> bit ) & 0x01 ) << 6 );

				addr = ( y + 7 ) * _gPaint.WidthByte + x;
				data |= ((( _gPaint.Image[ addr ] >> bit ) & 0x01 ) << 7 );

				Wire.write( data );
			}
			x += 1;
			Wire.endTransmission();
		}
		y += 8;
	}
}
/************************************************************************************************/
/*	機能概要：電子ﾍﾟｰﾊﾟｰ設定																	*/
/*	引数	：なし																				*/
/*	戻り値	：なし																				*/
/*	備考	：																					*/
/************************************************************************************************/
void _Epaper_Config()
{
	/* ---< ﾋﾟﾝｱｻｲﾝ >--- */
	pinMode( M_SPI_SCK,  OUTPUT );
	pinMode( M_SPI_DIN,  OUTPUT );
	pinMode( M_SPI_CS,   OUTPUT );
	pinMode( M_SPI_DC,   OUTPUT );
	pinMode( M_SPI_RST,  OUTPUT );
	pinMode( M_SPI_BUSY, INPUT  );

	digitalWrite( M_SPI_CS , GPIO_HIGH );
	digitalWrite( M_SPI_SCK, GPIO_LOW  );

	/* ---< SPI設定 >--- */
	_gSpi.begin( M_SPI_SCK, M_SPI_DOUT, M_SPI_DIN, M_SPI_CS );
	_gSpi.beginTransaction( SPISettings( 1000000, MSBFIRST, SPI_MODE0 ));
}
/************************************************************************************************/
/*	機能概要：電子ﾍﾟｰﾊﾟｰﾃﾞｨｽﾌﾟﾚｲ初期化															*/
/*	引数	：なし																				*/
/*	戻り値	：なし																				*/
/*	備考	：																					*/
/************************************************************************************************/
void _Epaper_InitDisplay()
{
	_Epaper_Reset();
	delay( _C_TIM_WAIT_100 );
	
	_Epape_WaitBusy();   
	_Epaper_CommandWrite( 0x12 );														// SW RESET
	_Epape_WaitBusy();   
	
	_Epaper_CommandWrite( 0x01 );														// Driver output Control      
	_Epaper_DataWrite(    0xf9 );
	_Epaper_DataWrite(    0x00 );
	_Epaper_DataWrite(    0x00 );
	
	_Epaper_CommandWrite( 0x11 );														// Data Entry mode setting      
	_Epaper_DataWrite(    0x03 );
	
	_Epaper_SetWindows( 0, 0, _C_SIZE_E_PAPER_WIDTH - 1, _C_SIZE_E_PAPER_HEIGHT - 1 );
	_Epaper_SetCursor(  0, 0 );
	
	_Epaper_CommandWrite( 0x3C );														// Border Wavefrom Control
	_Epaper_DataWrite(    0x05 );	
	
	_Epaper_CommandWrite( 0x21 );														// Display Update Control 1
	_Epaper_DataWrite(    0x00 );
	_Epaper_DataWrite(    0x80 );	
	
	_Epaper_CommandWrite( 0x18 );														// Temperature Sensor Control
	_Epaper_DataWrite(    0x80 );	
	
	_Epape_WaitBusy();
	_Epaper_SetLutByHost( WS_20_30_2IN13_V3 );
}
/************************************************************************************************/
/*	機能概要：電子ﾍﾟｰﾊﾟｰﾘｾｯﾄ(起床)																*/
/*	引数	：なし																				*/
/*	戻り値	：なし																				*/
/*	備考	：																					*/
/************************************************************************************************/
void _Epaper_Reset()
{
	digitalWrite( M_SPI_RST, GPIO_HIGH );
	delay( _C_TIM_WAIT_20 );
	digitalWrite( M_SPI_RST, GPIO_LOW  );
	delay( _C_TIM_WAIT_2  );
	digitalWrite( M_SPI_RST, GPIO_HIGH );
	delay( _C_TIM_WAIT_20 );
}
/************************************************************************************************/
/*	機能概要：電子ﾍﾟｰﾊﾟｰｽﾘｰﾌﾟ																	*/
/*	引数	：なし																				*/
/*	戻り値	：なし																				*/
/*	備考	：																					*/
/************************************************************************************************/
void _Epaper_Sleep()
{
	_Epaper_CommandWrite( 0x10 );
	_Epaper_DataWrite(    0x01 ); 
	delay( _C_TIM_WAIT_100 );
}
/************************************************************************************************/
/*	機能概要：電子ﾍﾟｰﾊﾟｰﾋﾞｼﾞｰ解放待ち															*/
/*	引数	：なし																				*/
/*	戻り値	：なし																				*/
/*	備考	：																					*/
/************************************************************************************************/
void _Epape_WaitBusy()
{
	Serial.println( "e-Paper busy" );
	while( true )
	{
		if( digitalRead( M_SPI_BUSY ) == GPIO_LOW )
		{
			break;
		}
		delay( _C_TIM_WAIT_10 );
	}
	delay( _C_TIM_WAIT_10 );

	Serial.println( "e-Paper released" );
}
/************************************************************************************************/
/*	機能概要：電子ﾍﾟｰﾊﾟｰﾃﾞｨｽﾌﾟﾚｲｳｲﾝﾄﾞｳｾｯﾄ														*/
/*	引数	：Sx X開始位置																		*/
/*			：Sy Y開始位置																		*/
/*			：Ex X終了位置																		*/
/*			：Ey Y終了位置																		*/
/*	戻り値	：なし																				*/
/*	備考	：																					*/
/************************************************************************************************/
void _Epaper_SetWindows( uint16_t Sx, uint16_t Sy, uint16_t Ex, uint16_t Ey )
{
	_Epaper_CommandWrite( 0x44 );														// X方向のRAMｳｲﾝﾄﾞｳｱﾄﾞﾚｽの開始/終了位置を指定
	_Epaper_DataWrite( ( Sx >> 3 ) & 0xff );
	_Epaper_DataWrite( ( Ex >> 3 ) & 0xff );
	
	_Epaper_CommandWrite( 0x45 );														// Y方向のRAMｳｲﾝﾄﾞｳｱﾄﾞﾚｽの開始/終了位置を指定
	_Epaper_DataWrite(  Sy & 0xff );
	_Epaper_DataWrite( ( Sy >> 8 ) & 0xff );
	_Epaper_DataWrite(  Ey & 0xff );
	_Epaper_DataWrite( ( Ey >> 8 ) & 0xff );
}
/************************************************************************************************/
/*	機能概要：電子ﾍﾟｰﾊﾟｰLUT設定																	*/
/*			：電子ﾍﾟｰﾊﾟｰLUTﾚｼﾞｽﾀ書込み															*/
/*	引数	：LUTﾃｰﾌﾞﾙ																			*/
/*	戻り値	：なし																				*/
/*	備考	：																					*/
/************************************************************************************************/
/*************************/
/*        LUT設定        */
/*************************/
void _Epaper_SetLutByHost( uint8_t* Lut )
{
	_Epaper_SetLut( (uint8_t *)Lut );													// LUTﾚｼﾞｽﾀ書込み
	_Epaper_CommandWrite( 0x3f );
	_Epaper_DataWrite( *(Lut+153) );
	_Epaper_CommandWrite( 0x03 );														// Gate Driving Voltage Control
	_Epaper_DataWrite( *(Lut+154) );
	_Epaper_CommandWrite( 0x04 );														// Source Driving Voltage Control
	_Epaper_DataWrite( *(Lut+155) );													// VSH
	_Epaper_DataWrite( *(Lut+156) );													// VSH2
	_Epaper_DataWrite( *(Lut+157) );													// VSL
	_Epaper_CommandWrite( 0x2c );														// VCOM
	_Epaper_DataWrite( *(Lut+158) );
}
/*************************/
/*    LUTﾚｼﾞｽﾀ書込み     */
/*************************/
void _Epaper_SetLut( uint8_t* Lut )
{
	_Epaper_CommandWrite( 0x32 );														// LUTﾚｼﾞｽﾀ書込み
	for( int index = 0; index < 153; index++ )
	{
		_Epaper_DataWrite( Lut[ index ] ); 
	}
	_Epape_WaitBusy();
}
/************************************************************************************************/
/*	機能概要：電子ﾍﾟｰﾊﾟｰｶｰｿﾙｾｯﾄ																	*/
/*	引数	：Sx X開始位置																		*/
/*			：Sy Y開始位置																		*/
/*	戻り値	：なし																				*/
/*	備考	：																					*/
/************************************************************************************************/
void _Epaper_SetCursor( uint16_t Sx, uint16_t Sy )
{
	_Epaper_CommandWrite( 0x4e );														// ｱﾄﾞﾚｽｶｳﾝﾀXの初期化
	_Epaper_DataWrite( Sx & 0xff );

	_Epaper_CommandWrite( 0x4f );														// ｱﾄﾞﾚｽｶｳﾝﾀYの初期化
	_Epaper_DataWrite(  Sy & 0xff );
	_Epaper_DataWrite( (Sy >> 8) & 0xff );
}
/************************************************************************************************/
/*	機能概要：電子ﾍﾟｰﾊﾟｰﾃﾞｨｽﾌﾟﾚｲﾘﾌﾚｯｼｭ															*/
/*			：部分的ﾃﾞｨｽﾌﾟﾚｲﾘﾌﾚｯｼｭ																*/
/*	引数	：なし																				*/
/*	戻り値	：なし																				*/
/*	備考	：																					*/
/************************************************************************************************/
/*************************/
/*    ﾃﾞｨｽﾌﾟﾚｲﾘﾌﾚｯｼｭ     */
/*************************/
void _Epaper_TurnOnDisplay()
{
	_Epaper_CommandWrite( 0x22 );														// Display Update Control 2
	_Epaper_DataWrite(    0xc7 );
	_Epaper_CommandWrite( 0x20 );														// Activate Display Update Sequence
	_Epape_WaitBusy();
}
/*************************/
/* 部分的ﾃﾞｨｽﾌﾟﾚｲﾘﾌﾚｯｼｭ  */
/*************************/
void _Epaper_TurnOnDisplayPartial()
{
	_Epaper_CommandWrite( 0x22 );														// Display Update Control 2
	_Epaper_DataWrite(    0x0f );														// fast:0x0c, quality:0x0f, 0xcf
	_Epaper_CommandWrite( 0x20 );														// Activate Display Update Sequence
	_Epape_WaitBusy();
}
/************************************************************************************************/
/*	機能概要：電子ﾍﾟｰﾊﾟｰﾃﾞｨｽﾌﾟﾚｲｸﾘｱ																*/
/*	引数	：なし																				*/
/*	戻り値	：なし																				*/
/*	備考	：																					*/
/************************************************************************************************/
void _Epaper_ClearDisplay()
{
	uint16_t width  = ( _C_SIZE_E_PAPER_WIDTH % 8 == 0 ) ? ( _C_SIZE_E_PAPER_WIDTH / 8 ) : ( _C_SIZE_E_PAPER_WIDTH / 8 + 1 );
	uint16_t height = _C_SIZE_E_PAPER_HEIGHT;

	// このｺﾏﾝﾄﾞの後、別のｺﾏﾝﾄﾞが書き込まれるまで、ﾃﾞｰﾀを白黒RAMに書き込む。書込み先は書込みと同時に進む
	_Epaper_CommandWrite( 0x24 );
	for( uint16_t j = 0; j < height; j++ ) 
	{
		for( uint16_t i = 0; i < width; i++) 
		{
			_Epaper_DataWrite( 0xff );													// 白書込み
		}
	}	
	
	_Epaper_CommandWrite( 0x26 );
	for( uint16_t j = 0; j < height; j++ ) 
	{
		for( uint16_t i = 0; i < width; i++ ) 
		{
			_Epaper_DataWrite( 0xff );													// 赤書込み???
		}
	}	
	_Epaper_TurnOnDisplay();
}
/************************************************************************************************/
/*	機能概要：電子ﾍﾟｰﾊﾟｰﾘﾌﾚｯｼｭ描画																*/
/*			：ﾍﾞｰｽｲﾒｰｼﾞﾘﾌﾚｯｼｭ描画																*/
/*			：部分的ﾘﾌﾚｯｼｭ描画																	*/
/*	引数	：Image ｲﾒｰｼﾞﾃﾞｰﾀ																	*/
/*	戻り値	：なし																				*/
/*	備考	：																					*/
/************************************************************************************************/
/*************************/
/*      ﾘﾌﾚｯｼｭ描画       */
/*************************/
void _Epaper_Display( uint8_t* Image )
{
	uint16_t width  = ( _C_SIZE_E_PAPER_WIDTH % 8 == 0 ) ? ( _C_SIZE_E_PAPER_WIDTH / 8 ) : ( _C_SIZE_E_PAPER_WIDTH / 8 + 1 );
	uint16_t height = _C_SIZE_E_PAPER_HEIGHT;
	
	_Epaper_CommandWrite( 0x24 );														// 白黒RAMへの書込み
	for( uint16_t j = 0; j < height; j++ ) 
	{
		for( uint16_t i = 0; i < width; i++ ) 
		{
			_Epaper_DataWrite( Image[ i + j * width ] );								// ｲﾒｰｼﾞ書込み
		}
	}	
	_Epaper_TurnOnDisplay();	
}
/*************************/
/*  ﾍﾞｰｽｲﾒｰｼﾞﾘﾌﾚｯｼｭ描画  */
/*************************/
void _Epaper_Display_Base( uint8_t* Image )
{  
	uint16_t width  = ( _C_SIZE_E_PAPER_WIDTH % 8 == 0 ) ? ( _C_SIZE_E_PAPER_WIDTH / 8 ) : ( _C_SIZE_E_PAPER_WIDTH / 8 + 1 );
	uint16_t height = _C_SIZE_E_PAPER_HEIGHT;
	
	_Epaper_CommandWrite( 0x24 );														// 白黒RAMへの書込み
	for( uint16_t j = 0; j < height; j++ ) 
	{
		for( uint16_t i = 0; i < width; i++ ) 
		{        
			_Epaper_DataWrite( Image[ i + j * width ] );								// ｲﾒｰｼﾞ書込み
		}
	}
	_Epaper_CommandWrite( 0x26 );														// 赤RAMへの書込み??
	for( uint16_t j = 0; j < height; j++ ) 
	{
		for( uint16_t i = 0; i < width; i++ ) 
		{
			_Epaper_DataWrite( Image[ i + j * width ] );								// ｲﾒｰｼﾞ書込み
		}
	}
	_Epaper_TurnOnDisplay();	
}
/*************************/
/*   部分的ﾘﾌﾚｯｼｭ描画    */
/*************************/
void _Epaper_Display_Partial( uint8_t* Image )
{
	uint16_t width  = ( _C_SIZE_E_PAPER_WIDTH % 8 == 0 ) ? ( _C_SIZE_E_PAPER_WIDTH / 8 ) : ( _C_SIZE_E_PAPER_WIDTH / 8 + 1 );
	uint16_t height = _C_SIZE_E_PAPER_HEIGHT;
	
	digitalWrite( M_SPI_RST, GPIO_LOW  );												// ﾘｾｯﾄ
	delay(1);
	digitalWrite( M_SPI_RST, GPIO_HIGH );

	_Epaper_SetLutByHost( WF_PARTIAL_2IN13_V3 );

	_Epaper_CommandWrite( 0x37 );														// Write Register for Display Option
	_Epaper_DataWrite(    0x00 );  
	_Epaper_DataWrite(    0x00 );  
	_Epaper_DataWrite(    0x00 );  
	_Epaper_DataWrite(    0x00 ); 
	_Epaper_DataWrite(    0x00 );  
	_Epaper_DataWrite(    0x40 );														// RAM Ping-Pong enable 
	_Epaper_DataWrite(    0x00 );  
	_Epaper_DataWrite(    0x00 );   
	_Epaper_DataWrite(    0x00 );  
	_Epaper_DataWrite(    0x00 );

	_Epaper_CommandWrite( 0x3C );														// Border Wavefrom Control
	_Epaper_DataWrite(    0x80 );	

	_Epaper_CommandWrite( 0x22 );														// Display Update Control 2
	_Epaper_DataWrite(    0xC0 );														// Enable clock and  Enable analog
	_Epaper_CommandWrite( 0x20 );														// Activate Display Update Sequence
	_Epape_WaitBusy();  
	
	_Epaper_SetWindows( 0, 0, _C_SIZE_E_PAPER_WIDTH - 1, _C_SIZE_E_PAPER_HEIGHT - 1 );
	_Epaper_SetCursor(  0, 0 );

	_Epaper_CommandWrite( 0x24 );														// 白黒RAMへの書込み
	for( uint16_t j = 0; j < height; j++ ) 
	{
		for( uint16_t i = 0; i < width; i++ ) 
		{
			_Epaper_DataWrite( Image[ i + j * width ] );								// ｲﾒｰｼﾞ書込み
		}
	}
	_Epaper_TurnOnDisplayPartial();
}
/************************************************************************************************/
/*	機能概要：電子ﾍﾟｰﾊﾟｰ1ﾊﾞｲﾄ送信																*/
/*	引数	：Data ｺﾏﾝﾄﾞ or ﾃﾞｰﾀ																*/
/*	戻り値	：なし																				*/
/*	備考	：																					*/
/************************************************************************************************/
void _Epaper_CommandWrite( uint8_t Data )
{
	digitalWrite( M_SPI_CS, GPIO_LOW );													// CS

	digitalWrite( M_SPI_DC, GPIO_LOW );													// Command送信
	_gSpi.write( Data );

	digitalWrite( M_SPI_CS, GPIO_HIGH );												// CS
}
void _Epaper_DataWrite( uint8_t Data )
{
	digitalWrite( M_SPI_CS, GPIO_LOW );													// CS

	digitalWrite( M_SPI_DC, GPIO_HIGH );												// Data送信
	_gSpi.write( Data );

	digitalWrite( M_SPI_CS, GPIO_HIGH );												// CS
}
/************************************************************************************************/
/*	機能概要：電子ﾍﾟｰﾊﾟｰ複数ﾊﾞｲﾄ送信															*/
/*	引数	：Data  ｺﾏﾝﾄﾞ or ﾃﾞｰﾀ																*/
/*			：Count 送信ﾊﾞｲﾄ数																	*/
/*	戻り値	：なし																				*/
/*	備考	：																					*/
/************************************************************************************************/
void _Epaper_CommandWriteBytes( uint8_t *Data, uint16_t Count )
{
	digitalWrite( M_SPI_CS, GPIO_LOW );													// CS

	digitalWrite( M_SPI_DC, GPIO_LOW );													// Command送信
	_gSpi.writeBytes( Data, Count );

	digitalWrite( M_SPI_CS, GPIO_HIGH );												// CS
}
void _Epaper_DataWriteBytes( uint8_t *Data, uint16_t Count )
{
	digitalWrite( M_SPI_CS, GPIO_LOW );													// CS

	digitalWrite( M_SPI_DC, GPIO_HIGH );												// Data送信
	_gSpi.writeBytes( Data, Count );

	digitalWrite( M_SPI_CS, GPIO_HIGH );												// CS
}
/************************************************************************************************/
/*	機能概要：ｲﾒｰｼﾞ構造体変数																	*/
/*	引数	：Image  ｲﾒｰｼﾞﾃﾞｰﾀへのﾎﾟｲﾝﾀ															*/
/*			：Width  ｲﾒｰｼﾞ幅																	*/
/*			：Height ｲﾒｰｼﾞ高																	*/
/*			：Rotate 回転																		*/
/*	戻り値	：なし																				*/
/*	備考	：																					*/
/************************************************************************************************/
void Paint_NewImage( uint8_t* Image, uint16_t Width, uint16_t Height, uint16_t Rotate )
{
	_gPaint.Image        = Image;
	_gPaint.WidthMemory  = Width;
	_gPaint.HeightMemory = Height;
	_gPaint.Rotate       = Rotate;
	_gPaint.Mirror       = enMirror_Non;
	_gPaint.WidthByte    = ( Width % 8 == 0 ) ? ( Width / 8 ) : ( Width / 8 + 1 );
	_gPaint.HeightByte   = Height;    
	
	if(( Rotate == C_DAT_ROTATE_0 ) || ( Rotate == C_DAT_ROTATE_180 ))
	{
		_gPaint.Width  = Width;
		_gPaint.Height = Height;
	}
	else
	{
		_gPaint.Width  = Height;
		_gPaint.Height = Width;
	}
}
/************************************************************************************************/
/*	機能概要：ｲﾒｰｼﾞｾｯﾄ																			*/
/*	引数	：Image ｲﾒｰｼﾞﾃﾞｰﾀへのﾎﾟｲﾝﾀ															*/
/*	戻り値	：なし																				*/
/*	備考	：																					*/
/************************************************************************************************/
void Paint_SelectImage( uint8_t* Image )
{
	_gPaint.Image = Image;
}
/************************************************************************************************/
/*	機能概要：回転ｾｯﾄ																			*/
/*	引数	：Rotate 回転角度																	*/
/*	戻り値	：なし																				*/
/*	備考	：																					*/
/************************************************************************************************/
void Paint_SetRotate( uint16_t Rotate )
{
	if(( Rotate == C_DAT_ROTATE_0 ) || ( Rotate == C_DAT_ROTATE_90 ) || ( Rotate == C_DAT_ROTATE_180 ) || ( Rotate == C_DAT_ROTATE_270 ))
	{
		_gPaint.Rotate = Rotate;
	}
}
/************************************************************************************************/
/*	機能概要：ﾐﾗｰｾｯﾄ																			*/
/*	引数	：Mirror ﾐﾗｰ																		*/
/*	戻り値	：なし																				*/
/*	備考	：																					*/
/************************************************************************************************/
void Paint_SetMirror( uint16_t Mirror )
{
	if(( Mirror == enMirror_Non ) || ( Mirror == enMirror_Horizontal ) || ( Mirror == enMirror_Vertical ) || ( Mirror == enMirror_Origin ))
	{
		_gPaint.Mirror = Mirror;
	}   
}
/************************************************************************************************/
/*	機能概要：ｲﾒｰｼﾞｸﾘｱ																			*/
/*	引数	：Color 塗潰し色(1ﾊﾞｲﾄ8ﾋﾟｸｾﾙ)														*/
/*	戻り値	：なし																				*/
/*	備考	：																					*/
/************************************************************************************************/
void Paint_Clear( uint16_t Color )
{
	for( uint16_t y = 0; y < _gPaint.HeightByte; y++ )
	{
		for( uint16_t x = 0; x < _gPaint.WidthByte; x++ )
		{
			_gPaint.Image[ x + y * _gPaint.WidthByte ] = Color;
		}
	}
}
/************************************************************************************************/
/*	機能概要：指定ﾋﾟｸｾﾙへの色ｾｯﾄ																*/
/*	引数	：Px    指定位置X																	*/
/*			：Py    指定位置Y																	*/
/*			：Color ｾｯﾄする色																	*/
/*	戻り値	：なし																				*/
/*	備考	：																					*/
/************************************************************************************************/
void Paint_SetPixel( uint16_t Px, uint16_t Py, uint16_t Color )
{
	if(( Px > _gPaint.Width ) || ( Py > _gPaint.Height ))
	{
		return;
	}      
	uint16_t x, y;
	switch( _gPaint.Rotate )
	{
		case C_DAT_ROTATE_0:
			x = Px;
			y = Py;  
			break;
		case C_DAT_ROTATE_90:
			x = _gPaint.WidthMemory  - Py - 1;
			y = Px;
			break;
		case C_DAT_ROTATE_180:
			x = _gPaint.WidthMemory  - Px - 1;
			y = _gPaint.HeightMemory - Py - 1;
			break;
		case C_DAT_ROTATE_270:
			x = Py;
			y = _gPaint.HeightMemory - Px - 1;
			break;
		default:
			return;
	}
	
	switch( _gPaint.Mirror )
	{
		case enMirror_Non:
			break;
		case enMirror_Horizontal:
			x = _gPaint.WidthMemory  - x - 1;
			break;
		case enMirror_Vertical:
			y = _gPaint.HeightMemory - y - 1;
			break;
		case enMirror_Origin:
			x = _gPaint.WidthMemory  - x - 1;
			y = _gPaint.HeightMemory - y - 1;
			break;
		default:
			return;
	}

	if(( x > _gPaint.WidthMemory ) || ( y > _gPaint.HeightMemory ))
	{
		return;
	}
	
	uint16_t addr = x / 8 + y * _gPaint.WidthByte;
	uint8_t  data = _gPaint.Image[ addr ];
	if( Color == _C_DAT_COLOR_BLACK )
	{
		_gPaint.Image[ addr ] = data & ~( 0x80 >> ( x % 8 ));
	}
	else
	{
		_gPaint.Image[ addr ] = data |  ( 0x80 >> ( x % 8 ));
	}
}
/************************************************************************************************/
/*	機能概要：指定ﾋﾟｸｾﾙへの色ｾｯﾄ																*/
/*	引数	：Sx    開始位置X																	*/
/*			：Sy    開始位置Y																	*/
/*			：Ex    終了位置Y																	*/
/*			：Ey    終了位置Y																	*/
/*			：Color ｾｯﾄする色																	*/
/*	戻り値	：なし																				*/
/*	備考	：																					*/
/************************************************************************************************/
void Paint_ClearWindows( uint16_t Sx, uint16_t Sy, uint16_t Ex, uint16_t Ey, uint16_t Color )
{
	for( uint16_t y = Sy; y < Ey; y++ )
	{
		for( uint16_t x = Sx; x < Ex; x++ )
		{
			Paint_SetPixel( x, y, Color );
		}
	}
}
/************************************************************************************************/
/*	機能概要：ﾄﾞｯﾄ描画																			*/
/*	引数	：Px    描画位置X																	*/
/*			：Py    描画位置Y																	*/
/*			：Color ｾｯﾄする色																	*/
/*			：Size  ﾄﾞｯﾄｻｲｽﾞ																	*/
/*			：Style ﾄﾞｯﾄｽﾀｲﾙ																	*/
/*	戻り値	：なし																				*/
/*	備考	：																					*/
/************************************************************************************************/
void Paint_DrawPoint( uint16_t Px, uint16_t Py, uint16_t Color, DOT_SIZE Size, DOT_STYLE Style )
{
	if(( Px > _gPaint.Width ) || ( Py > _gPaint.Height ))
	{
		return;
	}      

	/* ---< 角なしﾄﾞｯﾄ >--- */
	if( Style == enDot_Around )
	{
		for( int numX = 0; numX < 2 * Size - 1; numX++ )
		{
			for( int numY = 0; numY < 2 * Size - 1; numY++ )
			{
				if(( Px + numX - Size < 0 ) || ( Py + numY - Size < 0 ))
				{
					break;
				}
				Paint_SetPixel( Px + numX - Size, Py + numY - Size, Color );
			}
		}
	}
	/* ---< 角ありﾄﾞｯﾄ >--- */
	else
	{
		for( int numX = 0; numX < Size; numX++ )
		{
			for( int numY = 0; numY <  Size; numY++ )
			{
				Paint_SetPixel( Px + numX - 1, Py + numY - 1, Color );
			}
		}
	}
}
/************************************************************************************************/
/*	機能概要：ﾗｲﾝ描画																			*/
/*	引数	：Sx    開始位置X																	*/
/*			：Sy    開始位置Y																	*/
/*			：Ex    終了位置X																	*/
/*			：Ey    終了位置Y																	*/
/*			：Color ｾｯﾄする色																	*/
/*			：Width ﾗｲﾝ幅																		*/
/*			：Style ﾗｲﾝｽﾀｲﾙ																		*/
/*	戻り値	：なし																				*/
/*	備考	：																					*/
/************************************************************************************************/
void Paint_DrawLine( uint16_t Sx, uint16_t Sy, uint16_t Ex, uint16_t Ey, uint16_t Color, DOT_SIZE Width, LINE_STYLE Style )
{
	if(( Sx > _gPaint.Width ) || ( Sy > _gPaint.Height ) || ( Ex > _gPaint.Width ) || ( Ey > _gPaint.Height ))
	{
		return;
	}      

	uint16_t px = Sx;
	uint16_t py = Sy;
	int  dx     = (int)Ex - (int)Sx >= 0 ? Ex - Sx : Sx - Ex;
	int  dy     = (int)Ey - (int)Sy <= 0 ? Ey - Sy : Sy - Ey;
	int  diffX  = Sx < Ex ? 1 : -1;														// 増加方向
	int  diffY  = Sy < Ey ? 1 : -1;
	int  error  = dx + dy;																// 累積誤差
	char dotLen = 0;																	// 破線長

	for(;;)
	{
		/* ---< 破線描画 >--- */
		dotLen++;
		if(( Style == enLine_Dot ) && ( dotLen % 3 == 0 ))
		{
			Paint_DrawPoint( px, py, _gBackColor, Width, enDot_Around );
			dotLen = 0;
		} 
		/* ---< ﾗｲﾝ描画 >--- */
		else
		{
			Paint_DrawPoint( px, py, Color, Width, enDot_Around );
		}
		if( 2 * error >= dy )
		{
			if( px == Ex )
			{
				break;
			}
			error += dy;
			px    += diffX;
		}
		if( 2 * error <= dx )
		{
			if( py == Ey )
			{
				break;
			}
			error += dx;
			py    += diffY;
		}
	}
}
/************************************************************************************************/
/*	機能概要：矩形描画																			*/
/*	引数	：Sx    開始位置X																	*/
/*			：Sy    開始位置Y																	*/
/*			：Ex    終了位置X																	*/
/*			：Ey    終了位置Y																	*/
/*			：Color ｾｯﾄする色																	*/
/*			：Width ﾗｲﾝ幅																		*/
/*			：fill  塗潰し(true)																*/
/*	戻り値	：なし																				*/
/*	備考	：																					*/
/************************************************************************************************/
void Paint_DrawRectangle( uint16_t Sx, uint16_t Sy, uint16_t Ex, uint16_t Ey, uint16_t Color, DOT_SIZE Width, bool fill )
{
	if(( Sx > _gPaint.Width ) || ( Sy > _gPaint.Height ) || ( Ex > _gPaint.Width ) || ( Ey > _gPaint.Height ))
	{
		return;
	}      

	if( fill )
	{
		for( int y = Sy; y < Ey; y++ )
		{
			Paint_DrawLine( Sx, y, Ex, y, Color , Width, enLine_Solid );
		}
	}
	else
	{
		Paint_DrawLine( Sx, Sy, Ex, Sy, Color, Width, enLine_Solid );
		Paint_DrawLine( Sx, Sy, Sx, Ey, Color, Width, enLine_Solid );
		Paint_DrawLine( Ex, Ey, Ex, Sy, Color, Width, enLine_Solid );
		Paint_DrawLine( Ex, Ey, Sx, Ey, Color, Width, enLine_Solid );
	}
}
/************************************************************************************************/
/*	機能概要：矩形描画																			*/
/*	引数	：Cx     中心位置X																	*/
/*			：Cy     中心位置Y																	*/
/*			：Radius 半径																		*/
/*			：Color  ｾｯﾄする色																	*/
/*			：Width  ﾗｲﾝ幅																		*/
/*			：fill   塗潰し(true)																*/
/*	戻り値	：なし																				*/
/*	備考	：																					*/
/************************************************************************************************/
void Paint_DrawCircle( uint16_t Cx, uint16_t Cy, uint16_t Radius, uint16_t Color, DOT_SIZE Width, bool fill )
{
	if(( Cx > _gPaint.Width ) || ( Cy >= _gPaint.Height ))
	{
		return;
	}

	int16_t crtX  = 0;																// ( 0, R )を始点に描画
	int16_t crtY  = Radius;
	int16_t error = 3 - ( Radius << 1 );											// 累積誤差

	/* ---< 塗潰し円 >--- */
	if( fill )
	{
		while( crtX <= crtY )
		{	/* ---< 真円 >--- */
			for( int countY = crtX; countY <= crtY; countY++ )
			{
				Paint_DrawPoint( Cx + crtX,   Cy + countY, Color, enDot_1X1, enDot_Around );//1
				Paint_DrawPoint( Cx - crtX,   Cy + countY, Color, enDot_1X1, enDot_Around );//2
				Paint_DrawPoint( Cx - countY, Cy + crtX,   Color, enDot_1X1, enDot_Around );//3
				Paint_DrawPoint( Cx - countY, Cy - crtX,   Color, enDot_1X1, enDot_Around );//4
				Paint_DrawPoint( Cx - crtX,   Cy - countY, Color, enDot_1X1, enDot_Around );//5
				Paint_DrawPoint( Cx + crtX,   Cy - countY, Color, enDot_1X1, enDot_Around );//6
				Paint_DrawPoint( Cx + countY, Cy - crtX,   Color, enDot_1X1, enDot_Around );//7
				Paint_DrawPoint( Cx + countY, Cy + crtX,   Color, enDot_1X1, enDot_Around );
			}
			if( error < 0 )
			{
				error += 4 * crtX + 6;
			}
			else
			{
				error += 10 + 4 * ( crtX - crtY );
				crtY --;
			}
			crtX ++;
		}
	}
	/* ---< 中空円 >--- */
	else
	{
		while( crtX <= crtY )
		{
			Paint_DrawPoint( Cx + crtX, Cy + crtY, Color, Width, enDot_Around );	//1
			Paint_DrawPoint( Cx - crtX, Cy + crtY, Color, Width, enDot_Around );	//2
			Paint_DrawPoint( Cx - crtY, Cy + crtX, Color, Width, enDot_Around );	//3
			Paint_DrawPoint( Cx - crtY, Cy - crtX, Color, Width, enDot_Around );	//4
			Paint_DrawPoint( Cx - crtX, Cy - crtY, Color, Width, enDot_Around );	//5
			Paint_DrawPoint( Cx + crtX, Cy - crtY, Color, Width, enDot_Around );	//6
			Paint_DrawPoint( Cx + crtY, Cy - crtX, Color, Width, enDot_Around );	//7
			Paint_DrawPoint( Cx + crtY, Cy + crtX, Color, Width, enDot_Around );	//0

			if( error < 0 )
			{
				error += 4 * crtX + 6;
			}
			else
			{
				error += 10 + 4 * ( crtX - crtY );
				crtY --;
			}
			crtX ++;
		}
	}
}
/************************************************************************************************/
/*	機能概要：文字列描画																		*/
/*	引数	：Sx         描画開始位置X															*/
/*			：Sy         描画開始位置Y															*/
/*			：pString    文字列																	*/
/*			：Font       ﾌｫﾝﾄ																	*/
/*			：Foreground 前景色																	*/
/*			：Background 背景色																	*/
/*	戻り値	：なし																				*/
/*	備考	：																					*/
/************************************************************************************************/
void Paint_DrawString( uint16_t Sx, uint16_t Sy, const char* pString, FONT* Font, uint16_t Foreground, uint16_t Background )
{
	if(( Sx > _gPaint.Width ) || ( Sy >= _gPaint.Height ))
	{
		return;
	}

	uint16_t px = Sx;
	uint16_t py = Sy;
	while( *pString != '\0' )
	{
		/* ---< 描画ｴﾘｱをX方向で超えた場合 >--- */
		if( ( px + Font->Width ) > _gPaint.Width )
		{
			px = Sx;
			py += Font->Height;
		}

		/* ---< 描画ｴﾘｱをY方向で超えた場合 >--- */
		if( ( py  + Font->Height ) > _gPaint.Height )
		{
			px = Sx;
			py = Sy;
		}
		Paint_DrawChar( px, py, *pString, Font, Foreground, Background );

		/* ---< 次の文字更新 >--- */
		pString++;

		/* ---< 次の文字位置更新 >--- */
		px += Font->Width;
	}
}
/************************************************************************************************/
/*	機能概要：文字描画																			*/
/*	引数	：Px         描画位置X																*/
/*			：Py         描画位置Y																*/
/*			：Acsii_Char 文字																	*/
/*			：Font       ﾌｫﾝﾄ																	*/
/*			：Foreground 前景色																	*/
/*			：Background 背景色																	*/
/*	戻り値	：なし																				*/
/*	備考	：																					*/
/************************************************************************************************/
void Paint_DrawChar( uint16_t Px, uint16_t Py, const char Acsii_Char, FONT* Font, uint16_t Foreground, uint16_t Background )
{
	if(( Px > _gPaint.Width ) || ( Py >= _gPaint.Height ))
	{
		return;
	}

	uint32_t      offset = ( Acsii_Char - ' ' ) * Font->Height * ( Font->Width / 8 + ( Font->Width % 8 ? 1 : 0) );
	const uint8_t *ptr   = &( Font->Table[ offset ] );

	for( uint16_t fontY = 0; fontY < Font->Height; fontY++ )
	{
		for( uint16_t fontX = 0; fontX < Font->Width; fontX++ )
		{
			if( *ptr & ( 0x80 >> ( fontX % 8 )))
			{
				Paint_SetPixel( Px + fontX, Py + fontY, Foreground );
			}
			else
			{
				Paint_SetPixel( Px + fontX, Py + fontY, Background );
			}

			// 1文字は1ﾊﾞｲﾄ幅、2ﾊﾞｲﾄ、3ﾊﾞｲﾄ幅のﾃﾞｰﾀをもっているので、次のﾊﾞｲﾄに移る時にﾎﾟｲﾝﾀを更新する
			if( fontX % 8 == 7 )
			{
				ptr++;
			}
		}
		if( Font->Width % 8 != 0 )
		{
			ptr++;
		}
	}
}
/************************************************************************************************/
/*	機能概要：数値描画																			*/
/*	引数	：Sx         描画開始位置X															*/
/*			：Sy         描画開始位置Y															*/
/*			：Nummber    数値																	*/
/*			：Font       ﾌｫﾝﾄ																	*/
/*			：Foreground 前景色																	*/
/*			：Background 背景色																	*/
/*	戻り値	：なし																				*/
/*	備考	：																					*/
/************************************************************************************************/
void Paint_DrawNum( uint16_t Sx, uint16_t Sy, int32_t Nummber, FONT* Font, uint16_t Foreground, uint16_t Background )
{
	if(( Sx > _gPaint.Width ) || ( Sy >= _gPaint.Height ))
	{
		return;
	}

	int16_t Str_Pos = 0;
	int16_t Num_Pos = 0;
	uint8_t Str_Array[ _C_SIZE_NUM_TO_STR ] = { 0 };
	uint8_t Num_Array[ _C_SIZE_NUM_TO_STR ] = { 0 };
	uint8_t *pStr = Str_Array;

	/* ---< 数値を文字に変換 >--- */
	while( Nummber )
	{
		Num_Array[ Num_Pos ] = Nummber % 10 + '0';
		Num_Pos++;
		Nummber /= 10;
	}

	/* ---< 文字列を逆向きに変換 >--- */
	while( Num_Pos > 0 )
	{
		Str_Array[ Str_Pos ] = Num_Array[ Num_Pos - 1 ];
		Str_Pos++;
		Num_Pos--;
	}
 
	/* ---< 描画 >--- */
	Paint_DrawString( Sx, Sy, (const char*)pStr, Font, Foreground, Background );
}
/************************************************************************************************/
/*	機能概要：ﾋﾞｯﾄﾏｯﾌﾟ描画																		*/
/*	引数	：Image 描画ﾃﾞｰﾀ																	*/
/*	戻り値	：なし																				*/
/*	備考	：																					*/
/************************************************************************************************/
void Paint_DrawBitMap( const uint8_t* Image )
{
	for( int y = 0; y < _gPaint.HeightByte; y++ )
	{
		for( int x = 0; x < _gPaint.WidthByte; x++ )
		{
			uint32_t addr = x + y * _gPaint.WidthByte;
			_gPaint.Image[ addr ] = (uint8_t)Image[ addr ];
		}
	}
}
 
